using System;
using System.Windows.Forms;
using System.ComponentModel;

namespace CtrlLib
{
	/// <summary>
	/// Numeric data entry control
	/// </summary>
	public class NumEdit : TextBox
	{
		private NumEditType m_inpType;

		public enum NumEditType
		{
			Currency,
			Decimal,
			Single,
			Double,
			SmallInteger,
			Integer,
			LargeInteger,
			UInteger,
			ULargeInteger
		}

		public NumEdit()
		{
			// set default input type
			InputType = NumEditType.Integer;

			// NOTE: Existing context menu allows Paste command, with no known
			//	method to intercept. Only option is to reset to empty.
			//	(setting to Null doesn't work)
			this.ContextMenu = new ContextMenu();
		}

		[Description("Sets the numeric type allowed"), Category("Behavior")]
		public NumEditType InputType
		{
			get{return m_inpType;}
			set{m_inpType = value;}
		}

		public override string Text
		{
			get{return base.Text;}
			set
			{
				if(IsValid(value))
					base.Text = value;
			}
		}

		private bool IsValid(string val)
		{
			// this method validates the ENTIRE string
			//	not each character
			bool ret = true;

			// handle first char == '-'
			if(val.Equals("-"))
			{
				if( m_inpType == NumEditType.UInteger
					|| m_inpType == NumEditType.ULargeInteger )
				{
					return false;
				}
				else
				{
					return ret;
				}
			}

			// dissallow leading 0's
			if(/*val.Equals("0") ||*/ val.Equals("-0"))
				return false;

			if(val.Equals("") || val.Equals(String.Empty))
				return true;

			// parse into dataType, errors indicate invalid value
			// NOTE: parsing also validates data type min/max
			try
			{
				switch(m_inpType)
				{
					case NumEditType.Currency:
						decimal.Parse(val);
						int pos = val.IndexOf(".");
						if(pos != -1)
							ret = val.Substring(pos).Length <= 3;	// 2 decimals + "."
						break;
					case NumEditType.Single:
						float.Parse(val);
						break;
					case NumEditType.Double:
						double.Parse(val);
						break;
					case NumEditType.Decimal:
						decimal.Parse(val);
						break;
					case NumEditType.SmallInteger:
						short.Parse(val);
						break;
					case NumEditType.Integer:
						int.Parse(val);
						break;
					case NumEditType.LargeInteger:
						long.Parse(val);
						break;
					case NumEditType.UInteger:
						uint.Parse(val);
						break;
					case NumEditType.ULargeInteger:
						ulong.Parse(val);
						break;
					default:
						throw new ApplicationException();
				}
			}
			catch
			{
				ret = false;
			}
			return ret;
		}

		protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
		{
			// trap Ctrl-V paste and prevent invalid values
			//	return false to allow further processing
			if(keyData == (Keys)Shortcut.CtrlV)
			{
				IDataObject iData = Clipboard.GetDataObject();
 
				// assemble new string and check IsValid
				string newText;
				newText = base.Text.Substring(0, base.SelectionStart)
					+ (string)iData.GetData(DataFormats.Text)
					+ base.Text.Substring(base.SelectionStart + base.SelectionLength);

				// check if data to be pasted is convertable to inputType
				if(!IsValid(newText))
					return true;
			}
			return base.ProcessCmdKey(ref msg, keyData);
		}

		protected override void OnKeyPress(KeyPressEventArgs e)
		{
			// assemble new text with new KeyStroke
			//	and pass to validation routine.

			// NOTES;
			//	1) Delete key is NOT passed here
			//	2) control passed here after ProcessCmdKey() is run

			char c = e.KeyChar;
			if(!Char.IsControl(c))	// not sure about this?? nothing in docs about what is Control char??
			{
				// prevent spaces
				if(c.ToString() == " ")
				{
					e.Handled = true;
					return;
				}

				string newText = base.Text.Substring(0, base.SelectionStart)
					+ c.ToString() + base.Text.Substring(base.SelectionStart + base.SelectionLength);
				
				if(!IsValid(newText))
					e.Handled = true;
			}
			base.OnKeyPress(e);
		}
	}
}
